<script>
document.addEventListener('DOMContentLoaded', () => {
  const root = document.getElementById('top-slider');
  if (root) initSlider(root);
});

function initSlider(root) {
  const view  = root.querySelector('.s-viewport');
  const track = root.querySelector('.s-track');
  const prev  = root.querySelector('.s-prev');
  const next  = root.querySelector('.s-next');

  if (!view || !track) return;

  const speed = 400;
  const autoplayDelay = 5000;
  const breakpoints = [{w:0,perView:1,group:1},{w:480,perView:2,group:2},{w:768,perView:3,group:3},{w:1024,perView:3,group:3}];
  const swipeThreshold = 0.2;

  let originalSlides = Array.from(track.children).map(n => n.cloneNode(true));
  let perView = 3, group = 3, cloneN = 0;
  let slides = [], index = 0, slideW = 0, autoplayTimer = null;
  let startX = 0, currentX = 0, deltaX = 0, isDragging = false;

  function getBP() {
    const ww = window.innerWidth;
    return breakpoints.reduce((best, bp) => (ww >= bp.w ? bp : best), breakpoints[0]);
  }

  function translate(i, animate = true) {
    if (!animate) track.style.transition = 'none';
    const x = -i * slideW;
    track.style.transform = `translate3d(${x}px,0,0)`;
    if (!animate) {
      track.getBoundingClientRect();
      track.style.transition = `transform ${speed}ms ease`;
    }
  }

  function move(dir) {
    index += dir * group;
    translate(index, true);
  }

  function onEnd() {
    const real = slides.length - 2 * cloneN;
    const head = cloneN;
    const tail = head + real - 1;
    if (index < head) {
      index = tail - (head - index) + 1;
      translate(index, false);
    } else if (index > tail) {
      index = head + (index - tail) - 1;
      translate(index, false);
    }
  }

  function size() {
    slideW = view.clientWidth / perView;
    slides.forEach(s => s.style.width = slideW + 'px');
    track.style.width = (slideW * slides.length) + 'px';
    track.style.willChange = 'transform';
    track.style.transition = `transform ${speed}ms ease`;
    translate(index, false);
  }

  function startAutoplay() {
    stopAutoplay();
    if ((slides.length - 2*cloneN) <= perView) return;
    autoplayTimer = setInterval(() => move(1), autoplayDelay);
  }
  function stopAutoplay() {
    if (autoplayTimer) {
      clearInterval(autoplayTimer);
      autoplayTimer = null;
    }
  }

  function tweakTraueranzeigeImages() {
    track.querySelectorAll('.s-slide img').forEach(img => {
      const hay = [img.src, img.alt || '', img.title || '', img.getAttribute('data-src') || ''].join(' ').toLowerCase();
      if (hay.includes('traueranzeige')) {
        img.style.objectFit = 'contain';
        img.style.backgroundColor = '#000';
        img.style.objectPosition = 'top center';
        img.style.padding = '3% 10% 10% 10%';
      }
    });
  }

  function build() {
    track.removeEventListener('transitionend', onEnd);
    originalSlides = Array.from(track.children).map(n => n.cloneNode(true));
    track.innerHTML = '';
    slides = originalSlides.map(n => n.cloneNode(true));
    slides.forEach(n => track.appendChild(n));

    const bp = getBP();
    perView = bp.perView;
    group = bp.group;

    cloneN = Math.max(Math.ceil(perView), group);

    if (slides.length === 0) return;

    const first = slides.slice(0, cloneN).map(n => n.cloneNode(true));
    const last  = slides.slice(-cloneN).map(n => n.cloneNode(true));
    last.forEach(n => track.insertBefore(n, track.firstChild));
    first.forEach(n => track.appendChild(n));

    slides = Array.from(track.children);
    index = cloneN;

    size();
    tweakTraueranzeigeImages();

    track.addEventListener('transitionend', onEnd);

    stopAutoplay();
    startAutoplay();
  }

  function onTouchStart(e){
    const t = e.touches ? e.touches[0] : e;
    startX = currentX = t.clientX;
    deltaX = 0;
    isDragging = true;
    stopAutoplay();
    track.style.transition = 'none';
  }
  function onTouchMove(e){
    if (!isDragging) return;
    const t = e.touches ? e.touches[0] : e;
    currentX = t.clientX;
    deltaX = currentX - startX;
    const x = (-index * slideW) + deltaX;
    track.style.transform = `translate3d(${x}px,0,0)`;
    if (e.cancelable) e.preventDefault();
  }
  function onTouchEnd(){
    if (!isDragging) return;
    isDragging = false;
    track.style.transition = `transform ${speed}ms ease`;
    if (Math.abs(deltaX) > slideW * swipeThreshold) move(deltaX > 0 ? -1 : 1);
    else translate(index, true);
    deltaX = 0;
    startAutoplay();
  }

  if (prev) prev.addEventListener('click', () => { stopAutoplay(); move(-1); startAutoplay(); });
  if (next) next.addEventListener('click', () => { stopAutoplay(); move(1); startAutoplay(); });

  view.addEventListener('touchstart', onTouchStart, { passive: true });
  view.addEventListener('touchmove', onTouchMove, { passive: false });
  view.addEventListener('touchend', onTouchEnd);

  root.addEventListener('mouseenter', stopAutoplay);
  root.addEventListener('mouseleave', startAutoplay);
  document.addEventListener('visibilitychange', () => document.hidden ? stopAutoplay() : startAutoplay());

  window.addEventListener('resize', build, { passive: true });

  build();
}
</script>
